"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Users, Edit } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";

export default function ClassesPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [classes, setClasses] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isClassDialogOpen, setIsClassDialogOpen] = useState(false);
    const [isSectionDialogOpen, setIsSectionDialogOpen] = useState(false);
    const [selectedClassId, setSelectedClassId] = useState("");
    const [editingClass, setEditingClass] = useState<any>(null);
    const [editingSection, setEditingSection] = useState<any>(null);

    const [classFormData, setClassFormData] = useState({
        name: "",
        grade: "",
    });

    const [sectionFormData, setSectionFormData] = useState({
        name: "",
        capacity: "30",
    });

    useEffect(() => {
        fetchClasses();
    }, []);

    const fetchClasses = async () => {
        try {
            const res = await fetch('/api/classes');
            const data = await res.json();
            if (Array.isArray(data)) {
                setClasses(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load classes", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load classes", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const handleCreateClass = async () => {
        if (!classFormData.name || !classFormData.grade) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const url = editingClass ? `/api/classes/${editingClass.id}` : '/api/classes';
            const method = editingClass ? 'PUT' : 'POST';

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(classFormData),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingClass ? "Class updated successfully" : "Class created successfully" });
                setIsClassDialogOpen(false);
                setEditingClass(null);
                setClassFormData({ name: "", grade: "" });
                fetchClasses();
            } else {
                toast({ title: "Error", description: "Failed to save class", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save class", variant: "destructive" });
        }
    };

    const handleEditClass = (cls: any) => {
        setEditingClass(cls);
        setClassFormData({ name: cls.name, grade: cls.grade.toString() });
        setIsClassDialogOpen(true);
    };

    const handleCloseClassDialog = () => {
        setIsClassDialogOpen(false);
        setEditingClass(null);
        setClassFormData({ name: "", grade: "" });
    };

    const handleCreateSection = async () => {
        if (!sectionFormData.name) {
            toast({ title: "Error", description: "Please enter section name", variant: "destructive" });
            return;
        }

        try {
            const url = editingSection ? `/api/sections/${editingSection.id}` : '/api/sections';
            const method = editingSection ? 'PUT' : 'POST';

            const body = editingSection
                ? { name: sectionFormData.name, capacity: sectionFormData.capacity }
                : { ...sectionFormData, classId: selectedClassId };

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(body),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingSection ? "Section updated successfully" : "Section created successfully" });
                setIsSectionDialogOpen(false);
                setEditingSection(null);
                setSectionFormData({ name: "", capacity: "30" });
                fetchClasses();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to save section", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save section", variant: "destructive" });
        }
    };

    const handleEditSection = (section: any, classId: string) => {
        setEditingSection(section);
        setSelectedClassId(classId);
        setSectionFormData({ name: section.name, capacity: section.capacity.toString() });
        setIsSectionDialogOpen(true);
    };

    const handleCloseSectionDialog = () => {
        setIsSectionDialogOpen(false);
        setEditingSection(null);
        setSectionFormData({ name: "", capacity: "30" });
    };

    const handleDeleteClass = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}" and all its sections?`)) return;

        try {
            const res = await fetch(`/api/classes/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Class deleted successfully" });
                fetchClasses();
            } else {
                toast({ title: "Error", description: "Failed to delete class", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete class", variant: "destructive" });
        }
    };

    const handleDeleteSection = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}"?`)) return;

        try {
            const res = await fetch(`/api/sections/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Section deleted successfully" });
                fetchClasses();
            } else {
                toast({ title: "Error", description: "Failed to delete section", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete section", variant: "destructive" });
        }
    };

    const isAdmin = session?.user?.role === 'admin';

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Classes & Sections</h1>
                    <p className="text-muted-foreground">Manage classes and their sections</p>
                </div>
                {isAdmin && (
                    <Dialog open={isClassDialogOpen} onOpenChange={(open) => {
                        if (!open) handleCloseClassDialog();
                        else setIsClassDialogOpen(true);
                    }}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Class
                            </Button>
                        </DialogTrigger>
                        <DialogContent>
                            <DialogHeader>
                                <DialogTitle>{editingClass ? "Edit Class" : "Create New Class"}</DialogTitle>
                                <DialogDescription>{editingClass ? "Update class details" : "Add a new class/grade level"}</DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Class Name *</Label>
                                    <Input
                                        placeholder="e.g., Grade 10"
                                        value={classFormData.name}
                                        onChange={(e) => setClassFormData({ ...classFormData, name: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Grade Level *</Label>
                                    <Input
                                        type="number"
                                        placeholder="e.g., 10"
                                        value={classFormData.grade}
                                        onChange={(e) => setClassFormData({ ...classFormData, grade: e.target.value })}
                                    />
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={handleCloseClassDialog}>Cancel</Button>
                                <Button onClick={handleCreateClass}>{editingClass ? "Update Class" : "Create Class"}</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <div className="space-y-4">
                {loading ? (
                    Array.from({ length: 3 }).map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <div className="flex items-start justify-between">
                                    <div className="space-y-2">
                                        <Skeleton className="h-6 w-32" />
                                        <Skeleton className="h-4 w-24" />
                                    </div>
                                    <div className="flex gap-2">
                                        <Skeleton className="h-9 w-32 rounded-md" />
                                        <Skeleton className="h-9 w-9 rounded-md" />
                                        <Skeleton className="h-9 w-9 rounded-md" />
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-2">
                                    <Skeleton className="h-4 w-full" />
                                    <Skeleton className="h-4 w-3/4" />
                                </div>
                            </CardContent>
                        </Card>
                    ))
                ) : classes.map((cls) => (
                    <Card key={cls.id}>
                        <CardHeader>
                            <div className="flex items-start justify-between">
                                <div>
                                    <CardTitle>{cls.name}</CardTitle>
                                    <CardDescription className="flex items-center gap-2 mt-1">
                                        <Users className="h-4 w-4" />
                                        {cls.sections?.length || 0} sections
                                    </CardDescription>
                                </div>
                                <div className="flex gap-2">
                                    {isAdmin && (
                                        <>
                                            <Button
                                                variant="outline"
                                                size="sm"
                                                onClick={() => {
                                                    setSelectedClassId(cls.id);
                                                    setEditingSection(null);
                                                    setIsSectionDialogOpen(true);
                                                }}
                                            >
                                                <Plus className="h-4 w-4 mr-1" />
                                                Add Section
                                            </Button>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                onClick={() => handleEditClass(cls)}
                                            >
                                                <Edit className="h-4 w-4" />
                                            </Button>
                                            <Button
                                                variant="ghost"
                                                size="icon"
                                                onClick={() => handleDeleteClass(cls.id, cls.name)}
                                            >
                                                <Trash2 className="h-4 w-4 text-destructive" />
                                            </Button>
                                        </>
                                    )}
                                </div>
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="flex flex-wrap gap-2">
                                {cls.sections?.map((section: any) => (
                                    <div key={section.id} className="flex items-center gap-1">
                                        <Badge variant="secondary">
                                            {section.name} ({section.capacity} students)
                                        </Badge>
                                        {isAdmin && (
                                            <>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-6 w-6"
                                                    onClick={() => handleEditSection(section, cls.id)}
                                                >
                                                    <Edit className="h-3 w-3" />
                                                </Button>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-6 w-6"
                                                    onClick={() => handleDeleteSection(section.id, section.name)}
                                                >
                                                    <Trash2 className="h-3 w-3 text-destructive" />
                                                </Button>
                                            </>
                                        )}
                                    </div>
                                ))}
                                {(!cls.sections || cls.sections.length === 0) && (
                                    <p className="text-sm text-muted-foreground">No sections yet</p>
                                )}
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>

            {!loading && classes.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No classes found</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Class" to create one</p>
                        )}
                    </CardContent>
                </Card>
            )}

            <Dialog open={isSectionDialogOpen} onOpenChange={(open) => {
                if (!open) handleCloseSectionDialog();
                else setIsSectionDialogOpen(true);
            }}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>{editingSection ? "Edit Section" : "Create New Section"}</DialogTitle>
                        <DialogDescription>{editingSection ? "Update section details" : "Add a section to this class"}</DialogDescription>
                    </DialogHeader>
                    <div className="space-y-4 py-4">
                        <div className="space-y-2">
                            <Label>Section Name *</Label>
                            <Input
                                placeholder="e.g., Section A"
                                value={sectionFormData.name}
                                onChange={(e) => setSectionFormData({ ...sectionFormData, name: e.target.value })}
                            />
                        </div>
                        <div className="space-y-2">
                            <Label>Capacity</Label>
                            <Input
                                type="number"
                                placeholder="30"
                                value={sectionFormData.capacity}
                                onChange={(e) => setSectionFormData({ ...sectionFormData, capacity: e.target.value })}
                            />
                        </div>
                    </div>
                    <DialogFooter>
                        <Button variant="outline" onClick={handleCloseSectionDialog}>Cancel</Button>
                        <Button onClick={handleCreateSection}>{editingSection ? "Update Section" : "Create Section"}</Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
        </div>
    );
}
