"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Edit, DoorOpen, Building2 } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";

export default function RoomsPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [rooms, setRooms] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);
    const [editingRoom, setEditingRoom] = useState<any>(null);
    const [formData, setFormData] = useState({
        name: "",
        code: "",
        capacity: 30,
        building: "",
        floor: 1,
        roomType: "classroom",
        equipment: "",
        isActive: true,
    });

    useEffect(() => {
        fetchRooms();
    }, []);

    const fetchRooms = async () => {
        try {
            const res = await fetch('/api/rooms');
            const data = await res.json();
            if (Array.isArray(data)) {
                setRooms(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load rooms", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load rooms", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async () => {
        if (!formData.name || !formData.code) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const url = editingRoom ? `/api/rooms/${editingRoom.id}` : '/api/rooms';
            const method = editingRoom ? 'PUT' : 'POST';

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingRoom ? "Room updated successfully" : "Room created successfully" });
                setIsDialogOpen(false);
                setEditingRoom(null);
                setFormData({ name: "", code: "", capacity: 30, building: "", floor: 1, roomType: "classroom", equipment: "", isActive: true });
                fetchRooms();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to save room", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save room", variant: "destructive" });
        }
    };

    const handleEdit = (room: any) => {
        setEditingRoom(room);
        setFormData({
            name: room.name,
            code: room.code,
            capacity: room.capacity,
            building: room.building || "",
            floor: room.floor || 1,
            roomType: room.roomType,
            equipment: room.equipment || "",
            isActive: room.isActive,
        });
        setIsDialogOpen(true);
    };

    const handleCloseDialog = () => {
        setIsDialogOpen(false);
        setEditingRoom(null);
        setFormData({ name: "", code: "", capacity: 30, building: "", floor: 1, roomType: "classroom", equipment: "", isActive: true });
    };

    const handleDelete = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}"?`)) return;

        try {
            const res = await fetch(`/api/rooms/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Room deleted successfully" });
                fetchRooms();
            } else {
                toast({ title: "Error", description: "Failed to delete room", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete room", variant: "destructive" });
        }
    };

    const isAdmin = session?.user?.role === 'admin';

    const getRoomTypeColor = (type: string) => {
        switch (type) {
            case 'classroom': return 'bg-blue-500';
            case 'lab': return 'bg-purple-500';
            case 'gym': return 'bg-green-500';
            case 'auditorium': return 'bg-orange-500';
            case 'library': return 'bg-indigo-500';
            default: return 'bg-gray-500';
        }
    };

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Rooms</h1>
                    <p className="text-muted-foreground">Manage classrooms, labs, and other facilities</p>
                </div>
                {isAdmin && (
                    <Dialog open={isDialogOpen} onOpenChange={(open) => {
                        if (!open) handleCloseDialog();
                        else setIsDialogOpen(true);
                    }}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Room
                            </Button>
                        </DialogTrigger>
                        <DialogContent className="max-w-2xl">
                            <DialogHeader>
                                <DialogTitle>{editingRoom ? "Edit Room" : "Create New Room"}</DialogTitle>
                                <DialogDescription>
                                    {editingRoom ? "Update room details" : "Add a new room to the facility"}
                                </DialogDescription>
                            </DialogHeader>
                            <div className="grid grid-cols-2 gap-4 py-4">
                                <div className="space-y-2">
                                    <Label>Room Name *</Label>
                                    <Input
                                        placeholder="e.g., Room 101, Science Lab A"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Room Code *</Label>
                                    <Input
                                        placeholder="e.g., R101, SCLAB-A"
                                        value={formData.code}
                                        onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Capacity</Label>
                                    <Input
                                        type="number"
                                        placeholder="30"
                                        value={formData.capacity}
                                        onChange={(e) => setFormData({ ...formData, capacity: parseInt(e.target.value) || 30 })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Room Type</Label>
                                    <Select
                                        value={formData.roomType}
                                        onValueChange={(value) => setFormData({ ...formData, roomType: value })}
                                    >
                                        <SelectTrigger>
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="classroom">Classroom</SelectItem>
                                            <SelectItem value="lab">Laboratory</SelectItem>
                                            <SelectItem value="gym">Gymnasium</SelectItem>
                                            <SelectItem value="auditorium">Auditorium</SelectItem>
                                            <SelectItem value="library">Library</SelectItem>
                                            <SelectItem value="other">Other</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label>Building</Label>
                                    <Input
                                        placeholder="e.g., Main Building, Science Block"
                                        value={formData.building}
                                        onChange={(e) => setFormData({ ...formData, building: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Floor</Label>
                                    <Input
                                        type="number"
                                        placeholder="1"
                                        value={formData.floor}
                                        onChange={(e) => setFormData({ ...formData, floor: parseInt(e.target.value) || 1 })}
                                    />
                                </div>
                                <div className="space-y-2 col-span-2">
                                    <Label>Equipment</Label>
                                    <Input
                                        placeholder="e.g., Projector, Smartboard, Computers"
                                        value={formData.equipment}
                                        onChange={(e) => setFormData({ ...formData, equipment: e.target.value })}
                                    />
                                </div>
                                <div className="flex items-center space-x-2 col-span-2">
                                    <Switch
                                        checked={formData.isActive}
                                        onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                                    />
                                    <Label>Active</Label>
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={handleCloseDialog}>Cancel</Button>
                                <Button onClick={handleSubmit}>{editingRoom ? "Update Room" : "Create Room"}</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {loading ? (
                    Array.from({ length: 6 }).map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <div className="flex items-start justify-between">
                                    <div className="space-y-2">
                                        <Skeleton className="h-5 w-32" />
                                        <Skeleton className="h-4 w-24" />
                                    </div>
                                    <div className="flex gap-1">
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                    </div>
                                </div>
                            </CardHeader>
                        </Card>
                    ))
                ) : rooms.map((room) => (
                    <Card key={room.id} className={!room.isActive ? 'opacity-60' : ''}>
                        <CardHeader>
                            <div className="flex items-start justify-between">
                                <div className="space-y-2 flex-1">
                                    <div className="flex items-center gap-2">
                                        <DoorOpen className="h-4 w-4 text-muted-foreground" />
                                        <CardTitle className="text-lg">{room.name}</CardTitle>
                                    </div>
                                    <div className="flex flex-wrap gap-2">
                                        <Badge className={getRoomTypeColor(room.roomType) + " text-white"}>
                                            {room.code}
                                        </Badge>
                                        <Badge variant="outline">
                                            {room.capacity} seats
                                        </Badge>
                                        {!room.isActive && (
                                            <Badge variant="secondary">Inactive</Badge>
                                        )}
                                    </div>
                                    {room.building && (
                                        <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                            <Building2 className="h-3 w-3" />
                                            <span>{room.building}{room.floor ? `, Floor ${room.floor}` : ''}</span>
                                        </div>
                                    )}
                                    {room.equipment && (
                                        <p className="text-xs text-muted-foreground mt-1">
                                            Equipment: {room.equipment}
                                        </p>
                                    )}
                                </div>
                                {isAdmin && (
                                    <div className="flex gap-1">
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleEdit(room)}
                                        >
                                            <Edit className="h-4 w-4" />
                                        </Button>
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleDelete(room.id, room.name)}
                                        >
                                            <Trash2 className="h-4 w-4 text-destructive" />
                                        </Button>
                                    </div>
                                )}
                            </div>
                        </CardHeader>
                    </Card>
                ))}
            </div>

            {!loading && rooms.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <DoorOpen className="h-12 w-12 text-muted-foreground mb-4" />
                        <p className="text-muted-foreground">No rooms found</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Room" to create one</p>
                        )}
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
