"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Label } from "@/components/ui/label";
import { Plus, Calendar as CalendarIcon } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { useToast } from "@/hooks/use-toast";

const DAYS = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];

export default function RosterPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [classes, setClasses] = useState<any[]>([]);
    const [subjects, setSubjects] = useState<any[]>([]);
    const [teachers, setTeachers] = useState<any[]>([]);
    const [timeSlots, setTimeSlots] = useState<any[]>([]);
    const [schedules, setSchedules] = useState<any[]>([]);
    const [selectedSection, setSelectedSection] = useState<string>("");
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);

    const [newSchedule, setNewSchedule] = useState({
        teacherId: "",
        subjectId: "",
        timeSlotId: "",
        room: "",
    });

    useEffect(() => {
        fetchData();
    }, []);

    useEffect(() => {
        if (selectedSection) {
            fetchSchedules(selectedSection);
        }
    }, [selectedSection]);

    const fetchData = async () => {
        try {
            const [classesRes, subjectsRes, teachersRes, timeSlotsRes] = await Promise.all([
                fetch('/api/classes'),
                fetch('/api/subjects'),
                fetch('/api/teachers'),
                fetch('/api/timeslots'),
            ]);

            const [classesData, subjectsData, teachersData, timeSlotsData] = await Promise.all([
                classesRes.json(),
                subjectsRes.json(),
                teachersRes.json(),
                timeSlotsRes.json(),
            ]);

            setClasses(classesData);
            setSubjects(subjectsData);
            setTeachers(teachersData);
            setTimeSlots(timeSlotsData);
        } catch (error) {
            toast({ title: "Error", description: "Failed to load data", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const fetchSchedules = async (sectionId: string) => {
        try {
            const res = await fetch(`/api/schedules?sectionId=${sectionId}`);
            const data = await res.json();
            setSchedules(data);
        } catch (error) {
            toast({ title: "Error", description: "Failed to load schedules", variant: "destructive" });
        }
    };

    const handleCreateSchedule = async () => {
        if (!selectedSection || !newSchedule.teacherId || !newSchedule.subjectId || !newSchedule.timeSlotId) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const res = await fetch('/api/schedules', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...newSchedule,
                    sectionId: selectedSection,
                }),
            });

            if (res.ok) {
                toast({ title: "Success", description: "Schedule created successfully" });
                setIsDialogOpen(false);
                setNewSchedule({ teacherId: "", subjectId: "", timeSlotId: "", room: "" });
                fetchSchedules(selectedSection);
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error, variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to create schedule", variant: "destructive" });
        }
    };

    const handleDeleteSchedule = async (id: string) => {
        if (!confirm('Are you sure you want to delete this schedule?')) return;

        try {
            const res = await fetch(`/api/schedules?id=${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Schedule deleted successfully" });
                fetchSchedules(selectedSection);
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete schedule", variant: "destructive" });
        }
    };

    const getScheduleGrid = () => {
        const grid: any = {};
        const periods = [...new Set(timeSlots.map(ts => ts.period))].sort((a, b) => a - b);
        const days = [1, 2, 3, 4, 5]; // Monday to Friday

        days.forEach(day => {
            grid[day] = {};
            periods.forEach(period => {
                grid[day][period] = null;
            });
        });

        schedules.forEach(schedule => {
            const day = schedule.timeSlot.dayOfWeek;
            const period = schedule.timeSlot.period;
            if (grid[day] && grid[day][period] !== undefined) {
                grid[day][period] = schedule;
            }
        });

        return { grid, periods, days };
    };

    const { grid, periods, days } = selectedSection ? getScheduleGrid() : { grid: {}, periods: [], days: [] };

    const allSections = classes.flatMap(cls =>
        cls.sections.map((sec: any) => ({
            ...sec,
            className: cls.name
        }))
    );

    const isAdmin = session?.user?.role === 'admin';

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Roster Management</h1>
                    <p className="text-muted-foreground">Create and manage class schedules</p>
                </div>
                {isAdmin && selectedSection && (
                    <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Schedule
                            </Button>
                        </DialogTrigger>
                        <DialogContent>
                            <DialogHeader>
                                <DialogTitle>Create New Schedule</DialogTitle>
                                <DialogDescription>
                                    Add a new class schedule for the selected section
                                </DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Teacher</Label>
                                    <Select value={newSchedule.teacherId} onValueChange={(value) => setNewSchedule({ ...newSchedule, teacherId: value })}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select teacher" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {teachers.map((teacher) => (
                                                <SelectItem key={teacher.id} value={teacher.id}>
                                                    {teacher.name} {teacher.subject && `(${teacher.subject})`}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label>Subject</Label>
                                    <Select value={newSchedule.subjectId} onValueChange={(value) => setNewSchedule({ ...newSchedule, subjectId: value })}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select subject" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {subjects.map((subject) => (
                                                <SelectItem key={subject.id} value={subject.id}>
                                                    {subject.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label>Time Slot</Label>
                                    <Select value={newSchedule.timeSlotId} onValueChange={(value) => setNewSchedule({ ...newSchedule, timeSlotId: value })}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select time slot" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {timeSlots.map((slot) => (
                                                <SelectItem key={slot.id} value={slot.id}>
                                                    {DAYS[slot.dayOfWeek]} - Period {slot.period} ({slot.startTime} - {slot.endTime})
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label>Room (Optional)</Label>
                                    <Select value={newSchedule.room} onValueChange={(value) => setNewSchedule({ ...newSchedule, room: value })}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select room" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {['101', '102', '103', '201', '202', '203', 'LAB1', 'LAB2'].map((room) => (
                                                <SelectItem key={room} value={room}>Room {room}</SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={() => setIsDialogOpen(false)}>Cancel</Button>
                                <Button onClick={handleCreateSchedule}>Create Schedule</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <Card>
                <CardHeader>
                    <CardTitle>Select Section</CardTitle>
                    <CardDescription>Choose a class section to view and manage its schedule</CardDescription>
                </CardHeader>
                <CardContent>
                    {loading ? (
                        <Skeleton className="h-10 w-full md:w-[400px]" />
                    ) : (
                        <Select value={selectedSection} onValueChange={setSelectedSection}>
                            <SelectTrigger className="w-full md:w-[400px]">
                                <SelectValue placeholder="Select a section" />
                            </SelectTrigger>
                            <SelectContent>
                                {allSections.map((section) => (
                                    <SelectItem key={section.id} value={section.id}>
                                        {section.className} - {section.name}
                                    </SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                    )}
                </CardContent>
            </Card>

            {selectedSection && (
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <CalendarIcon className="h-5 w-5" />
                            Weekly Schedule
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        {loading ? (
                            <div className="space-y-3">
                                <Skeleton className="h-12 w-full" />
                                <Skeleton className="h-32 w-full" />
                                <Skeleton className="h-32 w-full" />
                                <Skeleton className="h-32 w-full" />
                            </div>
                        ) : (
                            <div className="overflow-x-auto">
                                <table className="w-full border-collapse">
                                    <thead>
                                        <tr>
                                            <th className="border p-3 bg-muted font-semibold">Period</th>
                                            {days.map(day => (
                                                <th key={day} className="border p-3 bg-muted font-semibold">
                                                    {DAYS[day]}
                                                </th>
                                            ))}
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {periods.map(period => {
                                            const slot = timeSlots.find(ts => ts.period === period);
                                            return (
                                                <tr key={period}>
                                                    <td className="border p-3 bg-muted/50 font-medium text-sm">
                                                        <div>Period {period}</div>
                                                        {slot && (
                                                            <div className="text-xs text-muted-foreground">
                                                                {slot.startTime} - {slot.endTime}
                                                            </div>
                                                        )}
                                                    </td>
                                                    {days.map(day => {
                                                        const schedule = grid[day]?.[period];
                                                        return (
                                                            <td key={`${day}-${period}`} className="border p-2 align-top">
                                                                {schedule ? (
                                                                    <div className="space-y-1">
                                                                        <Badge style={{ backgroundColor: schedule.subject.color }} className="text-white">
                                                                            {schedule.subject.code}
                                                                        </Badge>
                                                                        <div className="text-sm font-medium">{schedule.teacher.name}</div>
                                                                        {schedule.room && (
                                                                            <div className="text-xs text-muted-foreground">Room: {schedule.room}</div>
                                                                        )}
                                                                        {isAdmin && (
                                                                            <Button
                                                                                variant="ghost"
                                                                                size="sm"
                                                                                className="h-6 text-xs text-destructive hover:text-destructive"
                                                                                onClick={() => handleDeleteSchedule(schedule.id)}
                                                                            >
                                                                                Remove
                                                                            </Button>
                                                                        )}
                                                                    </div>
                                                                ) : (
                                                                    <div className="text-sm text-muted-foreground text-center py-4">-</div>
                                                                )}
                                                            </td>
                                                        );
                                                    })}
                                                </tr>
                                            );
                                        })}
                                    </tbody>
                                </table>
                            </div>
                        )}
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
