"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Edit, Calendar, CheckCircle2, Circle } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Switch } from "@/components/ui/switch";

export default function SeasonsPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [seasons, setSeasons] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);
    const [editingSeason, setEditingSeason] = useState<any>(null);

    const [formData, setFormData] = useState({
        name: "",
        startDate: "",
        endDate: "",
        isActive: false,
    });

    useEffect(() => {
        fetchSeasons();
    }, []);

    const fetchSeasons = async () => {
        try {
            const res = await fetch('/api/seasons');
            const data = await res.json();
            if (Array.isArray(data)) {
                setSeasons(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load seasons", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load seasons", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async () => {
        if (!formData.name || !formData.startDate || !formData.endDate) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const url = editingSeason ? `/api/seasons/${editingSeason.id}` : '/api/seasons';
            const method = editingSeason ? 'PUT' : 'POST';

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingSeason ? "Season updated successfully" : "Season created successfully" });
                setIsDialogOpen(false);
                setEditingSeason(null);
                setFormData({ name: "", startDate: "", endDate: "", isActive: false });
                fetchSeasons();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to save season", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save season", variant: "destructive" });
        }
    };

    const handleEdit = (season: any) => {
        setEditingSeason(season);
        setFormData({
            name: season.name,
            startDate: season.startDate.split('T')[0],
            endDate: season.endDate.split('T')[0],
            isActive: season.isActive,
        });
        setIsDialogOpen(true);
    };

    const handleCloseDialog = () => {
        setIsDialogOpen(false);
        setEditingSeason(null);
        setFormData({ name: "", startDate: "", endDate: "", isActive: false });
    };

    const handleDelete = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}"? This will also delete all associated time slots.`)) return;

        try {
            const res = await fetch(`/api/seasons/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Season deleted successfully" });
                fetchSeasons();
            } else {
                toast({ title: "Error", description: "Failed to delete season", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete season", variant: "destructive" });
        }
    };

    const handleToggleActive = async (season: any) => {
        try {
            const res = await fetch(`/api/seasons/${season.id}`, {
                method: 'PUT',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    name: season.name,
                    startDate: season.startDate,
                    endDate: season.endDate,
                    isActive: !season.isActive,
                }),
            });

            if (res.ok) {
                toast({ title: "Success", description: season.isActive ? "Season deactivated" : "Season activated" });
                fetchSeasons();
            } else {
                toast({ title: "Error", description: "Failed to update season", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to update season", variant: "destructive" });
        }
    };

    const isAdmin = session?.user?.role === 'admin';

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Seasons</h1>
                    <p className="text-muted-foreground">Manage different timetable seasons throughout the year</p>
                </div>
                {isAdmin && (
                    <Dialog open={isDialogOpen} onOpenChange={(open) => {
                        if (!open) handleCloseDialog();
                        else setIsDialogOpen(true);
                    }}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Season
                            </Button>
                        </DialogTrigger>
                        <DialogContent>
                            <DialogHeader>
                                <DialogTitle>{editingSeason ? "Edit Season" : "Create New Season"}</DialogTitle>
                                <DialogDescription>
                                    {editingSeason ? "Update season details" : "Add a new season with different time schedules"}
                                </DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Season Name *</Label>
                                    <Input
                                        placeholder="e.g., Winter, Summer, Spring"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Start Date *</Label>
                                    <Input
                                        type="date"
                                        value={formData.startDate}
                                        onChange={(e) => setFormData({ ...formData, startDate: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>End Date *</Label>
                                    <Input
                                        type="date"
                                        value={formData.endDate}
                                        onChange={(e) => setFormData({ ...formData, endDate: e.target.value })}
                                    />
                                </div>
                                <div className="flex items-center justify-between">
                                    <Label>Active Season</Label>
                                    <Switch
                                        checked={formData.isActive}
                                        onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                                    />
                                </div>
                                <p className="text-sm text-muted-foreground">
                                    Only one season can be active at a time. Active season determines which timetable is currently in use.
                                </p>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={handleCloseDialog}>Cancel</Button>
                                <Button onClick={handleSubmit}>{editingSeason ? "Update Season" : "Create Season"}</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                {loading ? (
                    Array.from({ length: 3 }).map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <Skeleton className="h-6 w-32" />
                                <Skeleton className="h-4 w-24 mt-2" />
                            </CardHeader>
                            <CardContent>
                                <Skeleton className="h-4 w-full" />
                            </CardContent>
                        </Card>
                    ))
                ) : seasons.map((season) => (
                    <Card key={season.id} className={season.isActive ? "border-primary" : ""}>
                        <CardHeader>
                            <div className="flex items-start justify-between">
                                <div className="flex items-center gap-2">
                                    {season.isActive ? (
                                        <CheckCircle2 className="h-5 w-5 text-primary" />
                                    ) : (
                                        <Circle className="h-5 w-5 text-muted-foreground" />
                                    )}
                                    <div>
                                        <CardTitle>{season.name}</CardTitle>
                                        {season.isActive && (
                                            <Badge variant="default" className="mt-1">Active</Badge>
                                        )}
                                    </div>
                                </div>
                                {isAdmin && (
                                    <div className="flex gap-1">
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleEdit(season)}
                                        >
                                            <Edit className="h-4 w-4" />
                                        </Button>
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleDelete(season.id, season.name)}
                                        >
                                            <Trash2 className="h-4 w-4 text-destructive" />
                                        </Button>
                                    </div>
                                )}
                            </div>
                        </CardHeader>
                        <CardContent className="space-y-3">
                            <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                <Calendar className="h-4 w-4" />
                                <span>
                                    {new Date(season.startDate).toLocaleDateString()} - {new Date(season.endDate).toLocaleDateString()}
                                </span>
                            </div>
                            <div className="text-sm text-muted-foreground">
                                {season._count?.timeSlots || 0} time slots configured
                            </div>
                            {isAdmin && !season.isActive && (
                                <Button
                                    variant="outline"
                                    size="sm"
                                    className="w-full"
                                    onClick={() => handleToggleActive(season)}
                                >
                                    Set as Active
                                </Button>
                            )}
                        </CardContent>
                    </Card>
                ))}
            </div>

            {!loading && seasons.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No seasons found</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Season" to create one</p>
                        )}
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
