"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Edit } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";

export default function SubjectsPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [subjects, setSubjects] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);
    const [editingSubject, setEditingSubject] = useState<any>(null);
    const [formData, setFormData] = useState({
        name: "",
        code: "",
        color: "#3B82F6",
    });

    useEffect(() => {
        fetchSubjects();
    }, []);

    const fetchSubjects = async () => {
        try {
            const res = await fetch('/api/subjects');
            const data = await res.json();
            if (Array.isArray(data)) {
                setSubjects(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load subjects", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load subjects", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async () => {
        if (!formData.name || !formData.code) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const url = editingSubject ? `/api/subjects/${editingSubject.id}` : '/api/subjects';
            const method = editingSubject ? 'PUT' : 'POST';

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(formData),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingSubject ? "Subject updated successfully" : "Subject created successfully" });
                setIsDialogOpen(false);
                setEditingSubject(null);
                setFormData({ name: "", code: "", color: "#3B82F6" });
                fetchSubjects();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to save subject", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save subject", variant: "destructive" });
        }
    };

    const handleEdit = (subject: any) => {
        setEditingSubject(subject);
        setFormData({
            name: subject.name,
            code: subject.code,
            color: subject.color,
        });
        setIsDialogOpen(true);
    };

    const handleCloseDialog = () => {
        setIsDialogOpen(false);
        setEditingSubject(null);
        setFormData({ name: "", code: "", color: "#3B82F6" });
    };

    const handleDelete = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}"?`)) return;

        try {
            const res = await fetch(`/api/subjects/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Subject deleted successfully" });
                fetchSubjects();
            } else {
                toast({ title: "Error", description: "Failed to delete subject", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete subject", variant: "destructive" });
        }
    };

    const isAdmin = session?.user?.role === 'admin';

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Subjects</h1>
                    <p className="text-muted-foreground">Manage school subjects and courses</p>
                </div>
                {isAdmin && (
                    <Dialog open={isDialogOpen} onOpenChange={(open) => {
                        if (!open) handleCloseDialog();
                        else setIsDialogOpen(true);
                    }}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Subject
                            </Button>
                        </DialogTrigger>
                        <DialogContent>
                            <DialogHeader>
                                <DialogTitle>{editingSubject ? "Edit Subject" : "Create New Subject"}</DialogTitle>
                                <DialogDescription>
                                    {editingSubject ? "Update subject details" : "Add a new subject to the curriculum"}
                                </DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Subject Name *</Label>
                                    <Input
                                        placeholder="e.g., Mathematics"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Subject Code *</Label>
                                    <Input
                                        placeholder="e.g., MATH"
                                        value={formData.code}
                                        onChange={(e) => setFormData({ ...formData, code: e.target.value.toUpperCase() })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Color</Label>
                                    <div className="flex gap-2">
                                        <Input
                                            type="color"
                                            value={formData.color}
                                            onChange={(e) => setFormData({ ...formData, color: e.target.value })}
                                            className="w-20 h-10"
                                        />
                                        <Input
                                            value={formData.color}
                                            onChange={(e) => setFormData({ ...formData, color: e.target.value })}
                                            placeholder="#3B82F6"
                                        />
                                    </div>
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={handleCloseDialog}>Cancel</Button>
                                <Button onClick={handleSubmit}>{editingSubject ? "Update Subject" : "Create Subject"}</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {loading ? (
                    Array.from({ length: 6 }).map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <div className="flex items-start justify-between">
                                    <div className="space-y-2">
                                        <Skeleton className="h-5 w-32" />
                                        <Skeleton className="h-6 w-20 rounded-full" />
                                    </div>
                                    <div className="flex gap-1">
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                    </div>
                                </div>
                            </CardHeader>
                        </Card>
                    ))
                ) : subjects.map((subject) => (
                    <Card key={subject.id}>
                        <CardHeader>
                            <div className="flex items-start justify-between">
                                <div className="space-y-1">
                                    <CardTitle className="text-lg">{subject.name}</CardTitle>
                                    <Badge style={{ backgroundColor: subject.color }} className="text-white">
                                        {subject.code}
                                    </Badge>
                                </div>
                                {isAdmin && (
                                    <div className="flex gap-1">
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleEdit(subject)}
                                        >
                                            <Edit className="h-4 w-4" />
                                        </Button>
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleDelete(subject.id, subject.name)}
                                        >
                                            <Trash2 className="h-4 w-4 text-destructive" />
                                        </Button>
                                    </div>
                                )}
                            </div>
                        </CardHeader>
                    </Card>
                ))}
            </div>

            {!loading && subjects.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No subjects found</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Subject" to create one</p>
                        )}
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
