"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Mail, BookOpen, Edit, Search } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";

export default function TeachersPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [teachers, setTeachers] = useState<any[]>([]);
    const [subjects, setSubjects] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);
    const [editingTeacher, setEditingTeacher] = useState<any>(null);
    const [searchQuery, setSearchQuery] = useState("");

    const [formData, setFormData] = useState({
        name: "",
        email: "",
        password: "",
        subjectIds: [] as string[],
    });

    useEffect(() => {
        Promise.all([fetchTeachers(), fetchSubjects()]).then(() => setLoading(false));
    }, []);

    const fetchTeachers = async () => {
        try {
            const res = await fetch('/api/teachers');
            const data = await res.json();
            if (Array.isArray(data)) {
                setTeachers(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load teachers", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load teachers", variant: "destructive" });
        }
    };

    const fetchSubjects = async () => {
        try {
            const res = await fetch('/api/subjects');
            const data = await res.json();
            if (Array.isArray(data)) {
                setSubjects(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load subjects", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load subjects", variant: "destructive" });
        }
    };

    const handleSubmit = async () => {
        if (!formData.name || !formData.email || (!editingTeacher && !formData.password)) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const url = editingTeacher ? `/api/teachers/${editingTeacher.id}` : '/api/teachers';
            const method = editingTeacher ? 'PUT' : 'POST';

            const body: any = {
                name: formData.name,
                email: formData.email,
                subjectIds: formData.subjectIds,
            };

            if (formData.password) {
                body.password = formData.password;
            }

            const res = await fetch(url, {
                method,
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(body),
            });

            if (res.ok) {
                toast({ title: "Success", description: editingTeacher ? "Teacher updated successfully" : "Teacher created successfully" });
                setIsDialogOpen(false);
                setEditingTeacher(null);
                setFormData({ name: "", email: "", password: "", subjectIds: [] });
                fetchTeachers();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to save teacher", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to save teacher", variant: "destructive" });
        }
    };

    const handleEdit = (teacher: any) => {
        setEditingTeacher(teacher);
        setFormData({
            name: teacher.name,
            email: teacher.email,
            password: "",
            subjectIds: teacher.subjects?.map((s: any) => s.id) || [],
        });
        setIsDialogOpen(true);
    };

    const handleCloseDialog = () => {
        setIsDialogOpen(false);
        setEditingTeacher(null);
        setFormData({ name: "", email: "", password: "", subjectIds: [] });
    };

    const handleDelete = async (id: string, name: string) => {
        if (!confirm(`Are you sure you want to delete "${name}"?`)) return;

        try {
            const res = await fetch(`/api/teachers/${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Teacher deleted successfully" });
                fetchTeachers();
            } else {
                toast({ title: "Error", description: "Failed to delete teacher", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete teacher", variant: "destructive" });
        }
    };

    const toggleSubject = (subjectId: string) => {
        setFormData(prev => ({
            ...prev,
            subjectIds: prev.subjectIds.includes(subjectId)
                ? prev.subjectIds.filter(id => id !== subjectId)
                : [...prev.subjectIds, subjectId]
        }));
    };

    const isAdmin = session?.user?.role === 'admin';

    const filteredTeachers = teachers.filter(teacher => {
        const query = searchQuery.toLowerCase();
        return (
            teacher.name.toLowerCase().includes(query) ||
            teacher.email.toLowerCase().includes(query) ||
            teacher.subjects?.some((subject: any) =>
                subject.name.toLowerCase().includes(query) ||
                subject.code.toLowerCase().includes(query)
            )
        );
    });

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Teachers</h1>
                    <p className="text-muted-foreground">Manage teaching staff and their subjects</p>
                </div>
                {isAdmin && (
                    <Dialog open={isDialogOpen} onOpenChange={(open) => {
                        if (!open) handleCloseDialog();
                        else setIsDialogOpen(true);
                    }}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Teacher
                            </Button>
                        </DialogTrigger>
                        <DialogContent className="max-w-md">
                            <DialogHeader>
                                <DialogTitle>{editingTeacher ? "Edit Teacher" : "Create New Teacher"}</DialogTitle>
                                <DialogDescription>{editingTeacher ? "Update teacher details" : "Add a new teacher to the system"}</DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Full Name *</Label>
                                    <Input
                                        placeholder="e.g., John Smith"
                                        value={formData.name}
                                        onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Email *</Label>
                                    <Input
                                        type="email"
                                        placeholder="teacher@school.edu"
                                        value={formData.email}
                                        onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Password {editingTeacher ? "(leave blank to keep current)" : "*"}</Label>
                                    <Input
                                        type="password"
                                        placeholder="••••••••"
                                        value={formData.password}
                                        onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Subjects</Label>
                                    <div className="flex flex-wrap gap-2 p-3 border rounded-md min-h-[80px]">
                                        {subjects.map(subject => (
                                            <Badge
                                                key={subject.id}
                                                variant={formData.subjectIds.includes(subject.id) ? "default" : "outline"}
                                                className="cursor-pointer"
                                                onClick={() => toggleSubject(subject.id)}
                                            >
                                                {subject.name}
                                            </Badge>
                                        ))}
                                        {subjects.length === 0 && (
                                            <p className="text-sm text-muted-foreground">No subjects available</p>
                                        )}
                                    </div>
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={handleCloseDialog}>Cancel</Button>
                                <Button onClick={handleSubmit}>{editingTeacher ? "Update Teacher" : "Create Teacher"}</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            <div className="relative">
                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                <Input
                    placeholder="Search by name, email, or subject..."
                    className="pl-9"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                {loading ? (
                    Array.from({ length: 6 }).map((_, i) => (
                        <Card key={i}>
                            <CardHeader>
                                <div className="flex items-start justify-between">
                                    <div className="space-y-2 flex-1">
                                        <Skeleton className="h-5 w-32" />
                                        <Skeleton className="h-4 w-48" />
                                    </div>
                                    <div className="flex gap-1">
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                        <Skeleton className="h-8 w-8 rounded-md" />
                                    </div>
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-2">
                                    <Skeleton className="h-4 w-20" />
                                    <div className="flex flex-wrap gap-1">
                                        <Skeleton className="h-6 w-16 rounded-full" />
                                        <Skeleton className="h-6 w-20 rounded-full" />
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    ))
                ) : filteredTeachers.map((teacher) => (
                    <Card key={teacher.id}>
                        <CardHeader>
                            <div className="flex items-start justify-between">
                                <div className="space-y-1">
                                    <CardTitle className="text-lg">{teacher.name}</CardTitle>
                                    <CardDescription className="flex items-center gap-1">
                                        <Mail className="h-3 w-3" />
                                        {teacher.email}
                                    </CardDescription>
                                </div>
                                {isAdmin && (
                                    <div className="flex gap-1">
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleEdit(teacher)}
                                        >
                                            <Edit className="h-4 w-4" />
                                        </Button>
                                        <Button
                                            variant="ghost"
                                            size="icon"
                                            onClick={() => handleDelete(teacher.id, teacher.name)}
                                        >
                                            <Trash2 className="h-4 w-4 text-destructive" />
                                        </Button>
                                    </div>
                                )}
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-2">
                                <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                    <BookOpen className="h-4 w-4" />
                                    <span>Subjects:</span>
                                </div>
                                <div className="flex flex-wrap gap-1">
                                    {teacher.subjects?.map((subject: any) => (
                                        <Badge key={subject.id} variant="secondary" style={{ backgroundColor: subject.color, color: 'white' }}>
                                            {subject.code}
                                        </Badge>
                                    ))}
                                    {(!teacher.subjects || teacher.subjects.length === 0) && (
                                        <p className="text-sm text-muted-foreground">No subjects assigned</p>
                                    )}
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                ))}
            </div>

            {!loading && teachers.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No teachers found</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Teacher" to create one</p>
                        )}
                    </CardContent>
                </Card>
            )}

            {!loading && teachers.length > 0 && filteredTeachers.length === 0 && (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No teachers match your search</p>
                        <p className="text-sm text-muted-foreground mt-2">Try a different search term</p>
                    </CardContent>
                </Card>
            )}
        </div>
    );
}
