"use client";

import { useState, useEffect } from "react";
import { useSession } from "next-auth/react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Plus, Trash2, Clock, Calendar } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";

const daysOfWeek = [
    { value: 1, label: "Monday" },
    { value: 2, label: "Tuesday" },
    { value: 3, label: "Wednesday" },
    { value: 4, label: "Thursday" },
    { value: 5, label: "Friday" },
    { value: 6, label: "Saturday" },
    { value: 0, label: "Sunday" },
];

export default function TimeSlotsPage() {
    const { data: session } = useSession();
    const { toast } = useToast();
    const [seasons, setSeasons] = useState<any[]>([]);
    const [selectedSeasonId, setSelectedSeasonId] = useState("");
    const [timeSlots, setTimeSlots] = useState<any[]>([]);
    const [loading, setLoading] = useState(true);
    const [isDialogOpen, setIsDialogOpen] = useState(false);

    const [formData, setFormData] = useState({
        dayOfWeek: "",
        startTime: "",
        endTime: "",
        period: "",
    });

    useEffect(() => {
        fetchSeasons();
    }, []);

    useEffect(() => {
        if (selectedSeasonId) {
            fetchTimeSlots();
        }
    }, [selectedSeasonId]);

    const fetchSeasons = async () => {
        try {
            const res = await fetch('/api/seasons');
            const data = await res.json();
            if (Array.isArray(data)) {
                setSeasons(data);
                // Auto-select active season
                const activeSeason = data.find((s: any) => s.isActive);
                if (activeSeason) {
                    setSelectedSeasonId(activeSeason.id);
                } else if (data.length > 0) {
                    setSelectedSeasonId(data[0].id);
                }
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load seasons", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const fetchTimeSlots = async () => {
        if (!selectedSeasonId) return;

        setLoading(true);
        try {
            const res = await fetch(`/api/timeslots?seasonId=${selectedSeasonId}`);
            const data = await res.json();
            if (Array.isArray(data)) {
                setTimeSlots(data);
            } else {
                toast({ title: "Error", description: data.error || "Failed to load time slots", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to load time slots", variant: "destructive" });
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async () => {
        if (!formData.dayOfWeek || !formData.startTime || !formData.endTime || !formData.period) {
            toast({ title: "Error", description: "Please fill all required fields", variant: "destructive" });
            return;
        }

        try {
            const res = await fetch('/api/timeslots', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    ...formData,
                    seasonId: selectedSeasonId,
                }),
            });

            if (res.ok) {
                toast({ title: "Success", description: "Time slot created successfully" });
                setIsDialogOpen(false);
                setFormData({ dayOfWeek: "", startTime: "", endTime: "", period: "" });
                fetchTimeSlots();
            } else {
                const error = await res.json();
                toast({ title: "Error", description: error.error || "Failed to create time slot", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to create time slot", variant: "destructive" });
        }
    };

    const handleDelete = async (id: string, period: number, day: number) => {
        const dayName = daysOfWeek.find(d => d.value === day)?.label;
        if (!confirm(`Are you sure you want to delete Period ${period} on ${dayName}?`)) return;

        try {
            const res = await fetch(`/api/timeslots?id=${id}`, { method: 'DELETE' });
            if (res.ok) {
                toast({ title: "Success", description: "Time slot deleted successfully" });
                fetchTimeSlots();
            } else {
                toast({ title: "Error", description: "Failed to delete time slot", variant: "destructive" });
            }
        } catch (error) {
            toast({ title: "Error", description: "Failed to delete time slot", variant: "destructive" });
        }
    };

    const isAdmin = session?.user?.role === 'admin';

    // Group time slots by day
    const groupedSlots = timeSlots.reduce((acc: any, slot) => {
        const day = slot.dayOfWeek;
        if (!acc[day]) acc[day] = [];
        acc[day].push(slot);
        return acc;
    }, {});

    return (
        <div className="space-y-6 animate-fade-in">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-heading font-bold">Time Slots</h1>
                    <p className="text-muted-foreground">Configure period timings for each season</p>
                </div>
                {isAdmin && selectedSeasonId && (
                    <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Time Slot
                            </Button>
                        </DialogTrigger>
                        <DialogContent>
                            <DialogHeader>
                                <DialogTitle>Create New Time Slot</DialogTitle>
                                <DialogDescription>Add a period timing for the selected season</DialogDescription>
                            </DialogHeader>
                            <div className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label>Day of Week *</Label>
                                    <Select value={formData.dayOfWeek} onValueChange={(value) => setFormData({ ...formData, dayOfWeek: value })}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select day" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {daysOfWeek.map((day) => (
                                                <SelectItem key={day.value} value={day.value.toString()}>
                                                    {day.label}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label>Period Number *</Label>
                                    <Input
                                        type="number"
                                        placeholder="e.g., 1, 2, 3"
                                        value={formData.period}
                                        onChange={(e) => setFormData({ ...formData, period: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>Start Time *</Label>
                                    <Input
                                        type="time"
                                        value={formData.startTime}
                                        onChange={(e) => setFormData({ ...formData, startTime: e.target.value })}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label>End Time *</Label>
                                    <Input
                                        type="time"
                                        value={formData.endTime}
                                        onChange={(e) => setFormData({ ...formData, endTime: e.target.value })}
                                    />
                                </div>
                            </div>
                            <DialogFooter>
                                <Button variant="outline" onClick={() => setIsDialogOpen(false)}>Cancel</Button>
                                <Button onClick={handleSubmit}>Create Time Slot</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                )}
            </div>

            {/* Season Selector */}
            <Card>
                <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                        <Calendar className="h-5 w-5" />
                        Select Season
                    </CardTitle>
                    <CardDescription>Choose which season's time slots to manage</CardDescription>
                </CardHeader>
                <CardContent>
                    <Select value={selectedSeasonId} onValueChange={setSelectedSeasonId}>
                        <SelectTrigger className="w-full md:w-[300px]">
                            <SelectValue placeholder="Select a season" />
                        </SelectTrigger>
                        <SelectContent>
                            {seasons.map((season) => (
                                <SelectItem key={season.id} value={season.id}>
                                    {season.name} {season.isActive && "(Active)"}
                                </SelectItem>
                            ))}
                        </SelectContent>
                    </Select>
                </CardContent>
            </Card>

            {/* Time Slots Grid */}
            {loading ? (
                <Card>
                    <CardContent className="py-6">
                        <Skeleton className="h-40 w-full" />
                    </CardContent>
                </Card>
            ) : !selectedSeasonId ? (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">Please create and select a season first</p>
                    </CardContent>
                </Card>
            ) : timeSlots.length === 0 ? (
                <Card>
                    <CardContent className="flex flex-col items-center justify-center py-12">
                        <p className="text-muted-foreground">No time slots configured for this season</p>
                        {isAdmin && (
                            <p className="text-sm text-muted-foreground mt-2">Click "Add Time Slot" to create one</p>
                        )}
                    </CardContent>
                </Card>
            ) : (
                <div className="space-y-4">
                    {daysOfWeek.map((day) => {
                        const slots = groupedSlots[day.value] || [];
                        if (slots.length === 0) return null;

                        return (
                            <Card key={day.value}>
                                <CardHeader>
                                    <CardTitle className="flex items-center gap-2">
                                        <Clock className="h-5 w-5" />
                                        {day.label}
                                    </CardTitle>
                                </CardHeader>
                                <CardContent>
                                    <div className="grid gap-2 md:grid-cols-2 lg:grid-cols-3">
                                        {slots
                                            .sort((a: any, b: any) => a.period - b.period)
                                            .map((slot: any) => (
                                                <div
                                                    key={slot.id}
                                                    className="flex items-center justify-between p-3 border rounded-lg"
                                                >
                                                    <div>
                                                        <div className="font-semibold">Period {slot.period}</div>
                                                        <div className="text-sm text-muted-foreground">
                                                            {slot.startTime} - {slot.endTime}
                                                        </div>
                                                    </div>
                                                    {isAdmin && (
                                                        <Button
                                                            variant="ghost"
                                                            size="icon"
                                                            onClick={() => handleDelete(slot.id, slot.period, slot.dayOfWeek)}
                                                        >
                                                            <Trash2 className="h-4 w-4 text-destructive" />
                                                        </Button>
                                                    )}
                                                </div>
                                            ))}
                                    </div>
                                </CardContent>
                            </Card>
                        );
                    })}
                </div>
            )}
        </div>
    );
}
